<?php
/**
 * Plugin Name: Panic-to-Plate Engine (Autopilot + Rescue)
 * Description: Panic-to-Plate Recipes + Autopilot (Decision Engine) + Rescue Mode shortcodes, with plan history + printable/downloadable grocery lists.
 * Version: 1.2.0
 * Author: You
 */

if (!defined('ABSPATH')) exit;

class P2P_Engine {
  const CPT = 'p2p_recipe';
  const USER_META_HISTORY = '_p2p_plan_history'; // array of plans (most recent first)

  public function __construct() {
    add_action('init', [$this, 'register_cpt_and_taxonomies']);
    add_action('init', [$this, 'handle_downloads']);
    add_action('add_meta_boxes', [$this, 'add_meta_boxes']);
    add_action('save_post', [$this, 'save_recipe_meta'], 10, 2);

    // Shortcodes (new)
    add_shortcode('panic_to_plate_decision_engine', [$this, 'shortcode_decision_engine']);
    add_shortcode('panic_to_plate_rescue_mode', [$this, 'shortcode_rescue_mode']);
    add_shortcode('panic_to_plate_my_plans', [$this, 'shortcode_my_plans']);

    // Backward-compatible aliases
    add_shortcode('decision_engine', [$this, 'shortcode_decision_engine']);
    add_shortcode('rescue_mode', [$this, 'shortcode_rescue_mode']);

    add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
  }

  public function enqueue_assets() {
    $css = "
      .p2p-wrap{max-width:980px;margin:20px auto;padding:16px}
      .p2p-card{border:1px solid #e5e5e5;border-radius:12px;padding:14px;margin:12px 0;background:#fff}
      .p2p-grid{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:12px}
      .p2p-grid-3{display:grid;grid-template-columns:repeat(3,minmax(0,1fr));gap:12px}
      .p2p-field label{display:block;font-weight:700;margin-bottom:6px}
      .p2p-field input[type=text], .p2p-field select{width:100%;padding:10px;border:1px solid #ddd;border-radius:10px}
      .p2p-btn{display:inline-block;padding:10px 14px;border-radius:10px;border:1px solid #15161A;background:#15161A;color:#fff;cursor:pointer;text-decoration:none}
      .p2p-btn.secondary{background:#fff;color:#15161A}
      .p2p-small{opacity:.8;font-size:.95em}
      .p2p-badges span{display:inline-block;border:1px solid #ddd;border-radius:999px;padding:4px 10px;margin:4px 6px 0 0;font-size:.85em}
      .p2p-actions{display:flex;gap:10px;flex-wrap:wrap;margin-top:10px}
      .p2p-hr{height:1px;background:#eee;margin:12px 0}
      details.p2p-details summary{cursor:pointer;font-weight:700}
      @media(max-width:700px){.p2p-grid,.p2p-grid-3{grid-template-columns:1fr}}
      @media print {.p2p-no-print{display:none !important}.p2p-wrap{max-width:100%;margin:0;padding:0}.p2p-card{border:none}}
    ";
    wp_register_style('p2p-engine-inline', false);
    wp_enqueue_style('p2p-engine-inline');
    wp_add_inline_style('p2p-engine-inline', $css);
  }

  public function register_cpt_and_taxonomies() {
    register_post_type(self::CPT, [
      'labels' => [
        'name'               => 'Panic-to-Plate Recipes',
        'singular_name'      => 'Panic-to-Plate Recipe',
        'add_new'            => 'Add Recipe',
        'add_new_item'       => 'Add New Recipe',
        'edit_item'          => 'Edit Recipe',
        'new_item'           => 'New Recipe',
        'view_item'          => 'View Recipe',
        'search_items'       => 'Search Recipes',
        'not_found'          => 'No recipes found',
        'not_found_in_trash' => 'No recipes found in Trash',
        'all_items'          => 'All Recipes',
        'menu_name'          => 'Panic-to-Plate',
        'name_admin_bar'     => 'P2P Recipe',
      ],
      'public' => true,
      'has_archive' => true,
      'menu_icon' => 'dashicons-carrot',
      'supports' => ['title','editor','thumbnail','excerpt'],
      'show_in_rest' => true,
      'rewrite' => ['slug' => 'recipes'],
    ]);

    $taxes = [
      'p2p_meal'        => ['Meal Type', 'Meal Types'],         // breakfast, lunch, dinner, snack
      'p2p_time'        => ['Time', 'Time'],                    // 5-10, 15-20, 30+
      'p2p_method'      => ['Method', 'Methods'],              // air fryer, one-pan, no-cook, etc
      'p2p_budget'      => ['Budget', 'Budget'],               // low, medium
      'p2p_protein'     => ['Protein', 'Proteins'],            // chicken, eggs, beans...
      'p2p_vibe'        => ['Vibe', 'Vibes'],                   // fresh, cozy, spicy...
      'p2p_rescue'      => ['Rescue Level', 'Rescue Levels'],   // rescue-1, rescue-2
      'p2p_pantry_item' => ['Pantry Item', 'Pantry Items'],     // eggs, tortillas, rice...
    ];

    foreach ($taxes as $slug => $labels) {
      register_taxonomy($slug, [self::CPT], [
        'labels' => [
          'name' => $labels[1],
          'singular_name' => $labels[0],
        ],
        'public' => true,
        'hierarchical' => false,
        'show_in_rest' => true,
        'rewrite' => ['slug' => $slug],
      ]);
    }
  }

  public function add_meta_boxes() {
    add_meta_box(
      'p2p_recipe_details',
      'Panic-to-Plate Recipe Details',
      [$this, 'render_meta_box'],
      self::CPT,
      'normal',
      'high'
    );
  }

  public function render_meta_box($post) {
    wp_nonce_field('p2p_save_recipe_meta', 'p2p_nonce');

    $ingredients = get_post_meta($post->ID, '_p2p_ingredients', true);
    $video_url   = get_post_meta($post->ID, '_p2p_video_url', true);

    echo '<p class="p2p-small">Tip: Ingredients should be one per line. This helps generate a simple grocery list.</p>';

    echo '<p><label><strong>Ingredients (one per line)</strong></label>';
    echo '<textarea style="width:100%;min-height:140px" name="p2p_ingredients">'.esc_textarea($ingredients).'</textarea></p>';

    echo '<p><label><strong>Video URL (Vimeo/YouTube link)</strong></label>';
    echo '<input style="width:100%" type="text" name="p2p_video_url" value="'.esc_attr($video_url).'" placeholder="https://..."></p>';
  }

  public function save_recipe_meta($post_id, $post) {
    if ($post->post_type !== self::CPT) return;
    if (!isset($_POST['p2p_nonce']) || !wp_verify_nonce($_POST['p2p_nonce'], 'p2p_save_recipe_meta')) return;
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $ingredients = isset($_POST['p2p_ingredients']) ? wp_kses_post($_POST['p2p_ingredients']) : '';
    $video_url   = isset($_POST['p2p_video_url']) ? esc_url_raw($_POST['p2p_video_url']) : '';

    update_post_meta($post_id, '_p2p_ingredients', $ingredients);
    update_post_meta($post_id, '_p2p_video_url', $video_url);
  }

  /* Downloads: grocery list (.txt)
   * Query: ?p2p_download_plan=<index>&p2p_type=grocery&p2p_nonce=<nonce>
   * index = 0 is most recent saved plan.
   */
  public function handle_downloads() {
    if (!isset($_GET['p2p_download_plan'], $_GET['p2p_type'], $_GET['p2p_nonce'])) return;

    $type = sanitize_text_field($_GET['p2p_type']);
    if ($type !== 'grocery') return;

    if (!is_user_logged_in()) {
      status_header(403);
      exit('You must be logged in to download this.');
    }

    $nonce = sanitize_text_field($_GET['p2p_nonce']);
    if (!wp_verify_nonce($nonce, 'p2p_download')) {
      status_header(403);
      exit('Invalid download token.');
    }

    $index = intval($_GET['p2p_download_plan']);
    $user_id = get_current_user_id();
    $history = get_user_meta($user_id, self::USER_META_HISTORY, true);
    if (!is_array($history) || !isset($history[$index])) {
      status_header(404);
      exit('Plan not found.');
    }

    $plan = $history[$index];
    $items = isset($plan['grocery']) && is_array($plan['grocery']) ? $plan['grocery'] : [];

    $filename = 'panic-to-plate-grocery-list-' . date('Y-m-d') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="'.$filename.'"');

    echo "Panic-to-Plate Grocery List\n";
    echo "Generated: " . gmdate('Y-m-d H:i') . " UTC\n\n";
    foreach ($items as $it) echo "- " . $it . "\n";
    exit;
  }

  /* Decision Engine (Autopilot)
   * Shortcode: [panic_to_plate_decision_engine] (aliases: [decision_engine])
   */
  public function shortcode_decision_engine($atts) {
    $atts = shortcode_atts([
      'max_candidates' => 80,
    ], $atts);

    $out = '<div class="p2p-wrap"><div class="p2p-card">';
    $out .= '<h2>Autopilot Plan</h2><p class="p2p-small">Answer a few questions and get a weekly plan built from your recipe library.</p>';

    $out .= '<form method="post">';
    $out .= wp_nonce_field('p2p_decision_engine', 'p2p_de_nonce', true, false);

    $out .= '<div class="p2p-grid">';
    $out .= $this->select_field('time', 'Weeknight time', [
      '' => 'Any',
      '5-10' => '5–10 min',
      '15-20' => '15–20 min',
      '30+' => '30+ min',
    ]);

    $out .= $this->select_field('method', 'Preferred method', [
      '' => 'Any',
      'no-cook' => 'No-cook',
      'microwave' => 'Microwave',
      'air-fryer' => 'Air fryer',
      'one-pan' => 'One-pan',
      'sheet-pan' => 'Sheet pan',
      'slow-cooker' => 'Slow cooker',
      'stovetop' => 'Stovetop',
    ]);

    $out .= $this->select_field('budget', 'Budget', [
      '' => 'Any',
      'low' => 'Low',
      'medium' => 'Medium',
    ]);

    $out .= $this->select_field('protein', 'Preferred protein', [
      '' => 'Any',
      'chicken' => 'Chicken',
      'turkey' => 'Turkey',
      'beef' => 'Beef',
      'eggs' => 'Eggs',
      'beans' => 'Beans',
      'tofu' => 'Tofu',
      'fish' => 'Fish',
    ]);
    $out .= '</div>';

    $out .= '<div class="p2p-grid">';
    $out .= $this->select_field('cook_nights', 'Cook dinners per week', [
      '3' => '3 nights',
      '4' => '4 nights',
      '5' => '5 nights',
      '6' => '6 nights',
      '7' => '7 nights',
    ], '5');

    $out .= $this->select_field('leftovers', 'Use leftovers for lunch?', [
      'yes' => 'Yes',
      'no'  => 'No',
    ], 'yes');
    $out .= '</div>';

    $out .= '<p class="p2p-field"><label>Foods you hate (comma-separated)</label>';
    $out .= '<input type="text" name="hate" placeholder="mushrooms, olives, ..." value="'.esc_attr($this->postv('hate')).'"></p>';

    $out .= '<div class="p2p-actions p2p-no-print">';
    $out .= '<button class="p2p-btn" type="submit" name="p2p_generate" value="1">Generate my week</button>';
    $out .= '<button class="p2p-btn secondary" type="submit" name="p2p_generate" value="1">Regenerate</button>';
    $out .= '</div>';

    $out .= '</form></div>';

    if (isset($_POST['p2p_generate']) && $this->verify_nonce('p2p_de_nonce', 'p2p_decision_engine')) {
      $prefs = [
        'time'       => sanitize_text_field($this->postv('time')),
        'method'     => sanitize_text_field($this->postv('method')),
        'budget'     => sanitize_text_field($this->postv('budget')),
        'protein'    => sanitize_text_field($this->postv('protein')),
        'cook_nights'=> max(3, min(7, intval($this->postv_default('cook_nights', '5')))),
        'leftovers'  => sanitize_text_field($this->postv_default('leftovers', 'yes')),
        'hate'       => array_filter(array_map('trim', explode(',', strtolower(sanitize_text_field($this->postv('hate')))))),
      ];

      $plan = $this->build_weekly_plan($prefs, (int)$atts['max_candidates']);

      $saved_index = null;
      if (is_user_logged_in()) {
        $saved_index = $this->save_plan_history(get_current_user_id(), $plan, $prefs);
      }

      $out .= '<div class="p2p-card">';
      $out .= '<h3>Your Weekly Plan</h3>';
      $out .= $plan['html'];
      $out .= '</div>';

      $out .= '<div class="p2p-card" id="p2p-grocery">';
      $out .= '<h3>Grocery List</h3>';
      $out .= '<div class="p2p-actions p2p-no-print">';
      $out .= '<button class="p2p-btn secondary" type="button" onclick="window.print()">Print Grocery List</button>';

      if ($saved_index !== null) {
        $dl = add_query_arg([
          'p2p_download_plan' => $saved_index,
          'p2p_type' => 'grocery',
          'p2p_nonce' => wp_create_nonce('p2p_download'),
        ], home_url('/'));
        $out .= '<a class="p2p-btn" href="'.esc_url($dl).'">Download .txt</a>';
      } else {
        $out .= '<span class="p2p-small">Log in to enable downloads + plan history.</span>';
      }

      $out .= '</div>';
      $out .= $plan['grocery_html'];
      $out .= '</div>';
    }

    $out .= '</div>';
    return $out;
  }

  private function save_plan_history($user_id, $plan, $prefs) {
    $history = get_user_meta($user_id, self::USER_META_HISTORY, true);
    if (!is_array($history)) $history = [];

    $entry = [
      'created_at' => time(),
      'prefs'      => $prefs,
      'days'       => $plan['days_ids'],
      'grocery'    => $plan['grocery_items'],
    ];

    array_unshift($history, $entry);
    $history = array_slice($history, 0, 12);

    update_user_meta($user_id, self::USER_META_HISTORY, $history);
    return 0;
  }

  private function build_weekly_plan($prefs, $max_candidates = 80) {
    $tax_query = ['relation' => 'AND'];

    if (!empty($prefs['time']))    $tax_query[] = ['taxonomy'=>'p2p_time','field'=>'slug','terms'=>[$prefs['time']]];
    if (!empty($prefs['method']))  $tax_query[] = ['taxonomy'=>'p2p_method','field'=>'slug','terms'=>[$prefs['method']]];
    if (!empty($prefs['budget']))  $tax_query[] = ['taxonomy'=>'p2p_budget','field'=>'slug','terms'=>[$prefs['budget']]];
    if (!empty($prefs['protein'])) $tax_query[] = ['taxonomy'=>'p2p_protein','field'=>'slug','terms'=>[$prefs['protein']]];

    $args = [
      'post_type'      => self::CPT,
      'posts_per_page' => $max_candidates,
      'post_status'    => 'publish',
      'orderby'        => 'rand',
    ];
    if (count($tax_query) > 1) $args['tax_query'] = $tax_query;

    $q = new WP_Query($args);
    $candidates = $q->posts;

    $filtered = [];
    foreach ($candidates as $p) {
      $title = strtolower($p->post_title);
      $ings  = strtolower((string)get_post_meta($p->ID, '_p2p_ingredients', true));
      $bad = false;
      foreach ($prefs['hate'] as $h) {
        if ($h === '') continue;
        if (strpos($title, $h) !== false || strpos($ings, $h) !== false) { $bad = true; break; }
      }
      if (!$bad) $filtered[] = $p;
    }

    $pick = function($meal_slug, $n) use ($filtered) {
      $bucket = [];
      foreach ($filtered as $p) if (has_term($meal_slug, 'p2p_meal', $p)) $bucket[] = $p;
      shuffle($bucket);
      return array_slice($bucket, 0, $n);
    };

    $rescue = $this->pick_rescue(2);

    $breakfasts = $pick('breakfast', 2);
    $lunches    = $pick('lunch', 2);

    $cook_nights = intval($prefs['cook_nights']);
    $leftovers   = ($prefs['leftovers'] === 'yes');

    $dinner_recipe_count = max(0, min(7, $cook_nights));
    $dinners = $pick('dinner', max(1, $dinner_recipe_count));

    $fallback = function(&$arr, $need) use ($filtered) {
      while (count($arr) < $need && count($filtered) > 0) $arr[] = $filtered[array_rand($filtered)];
    };

    $fallback($breakfasts, 2);
    $fallback($lunches, 2);
    $fallback($dinners, max(1, $dinner_recipe_count));

    $days = ['Mon','Tue','Wed','Thu','Fri','Sat','Sun'];

    $rescue_days = [2, 4]; // Wed + Fri
    $planned_dinners_total = min(7, $cook_nights + 2);
    $flex_count = max(0, 7 - $planned_dinners_total);

    $flex_days = [];
    if ($flex_count > 0) {
      $cands = [5,6,1,3,0,2,4];
      $flex_days = array_slice($cands, 0, $flex_count);
    }

    $html = '<div class="p2p-grid">';
    $grocery = [];
    $days_ids = [];

    for ($i=0; $i<7; $i++) {
      $b = $breakfasts[$i % max(1,count($breakfasts))] ?? null;

      $l = null;
      $lunch_label = 'Lunch';
      if ($leftovers && $i > 0) $lunch_label = 'Lunch (Leftovers)';
      else $l = $lunches[$i % max(1,count($lunches))] ?? null;

      $use_rescue = in_array($i, $rescue_days, true);
      $use_flex   = in_array($i, $flex_days, true);

      $d = null;
      $d_label = 'Dinner';
      if ($use_flex) $d_label = 'Dinner (Flex Night)';
      elseif ($use_rescue) { $d = $rescue[($i === $rescue_days[0]) ? 0 : 1] ?? null; $d_label = 'Dinner (Rescue)'; }
      else $d = $dinners[$i % max(1,count($dinners))] ?? null;

      $html .= '<div class="p2p-card">';
      $html .= '<h4>'.$days[$i].'</h4>';
      $html .= $this->plan_item('Breakfast', $b, $grocery);

      if ($leftovers && $i > 0) $html .= '<p><strong>'.esc_html($lunch_label).':</strong> Leftovers from dinner</p>';
      else $html .= $this->plan_item($lunch_label, $l, $grocery);

      if ($use_flex) $html .= '<p><strong>'.esc_html($d_label).':</strong> Flex night (eat out / leftovers / snack plate)</p>';
      else $html .= $this->plan_item($d_label, $d, $grocery);

      $html .= '</div>';

      $days_ids[] = [
        'b' => $b ? $b->ID : 0,
        'l' => ($leftovers && $i > 0) ? 0 : ($l ? $l->ID : 0),
        'd' => ($use_flex ? -1 : ($d ? $d->ID : 0)),
      ];
    }

    $html .= '</div>';

    $grocery = array_unique(array_filter(array_map('trim', $grocery)));
    sort($grocery);

    $grocery_html = '<ul>';
    foreach ($grocery as $item) $grocery_html .= '<li>'.esc_html($item).'</li>';
    $grocery_html .= '</ul>';

    return [
      'html' => $html,
      'grocery_html' => $grocery_html,
      'grocery_items' => $grocery,
      'days_ids' => $days_ids,
    ];
  }

  private function plan_item($label, $post, &$grocery) {
    if (!$post) return '<p><strong>'.esc_html($label).':</strong> <em>Add more recipes to your library.</em></p>';

    $permalink = get_permalink($post);
    $video = get_post_meta($post->ID, '_p2p_video_url', true);

    $ings = (string)get_post_meta($post->ID, '_p2p_ingredients', true);
    foreach (preg_split('/\R+/', $ings) as $line) {
      $line = trim(wp_strip_all_tags($line));
      if ($line !== '') $grocery[] = $line;
    }

    $badges = $this->get_badges($post->ID);

    $html = '<p><strong>'.esc_html($label).':</strong> <a href="'.esc_url($permalink).'">'.esc_html(get_the_title($post)).'</a></p>';
    if ($badges) $html .= '<div class="p2p-badges">'.$badges.'</div>';
    if ($video) $html .= '<p class="p2p-small"><a href="'.esc_url($video).'" target="_blank" rel="noopener">Watch video</a></p>';

    return $html;
  }

  private function get_badges($post_id) {
    $taxes = ['p2p_time','p2p_method','p2p_budget','p2p_protein','p2p_vibe','p2p_rescue'];
    $out = '';
    foreach ($taxes as $tax) {
      $terms = get_the_terms($post_id, $tax);
      if (!empty($terms) && !is_wp_error($terms)) $out .= '<span>'.esc_html($terms[0]->name).'</span>';
    }
    return $out;
  }

  private function pick_rescue($n) {
    $q = new WP_Query([
      'post_type'      => self::CPT,
      'posts_per_page' => $n,
      'post_status'    => 'publish',
      'orderby'        => 'rand',
      'tax_query'      => [[ 'taxonomy'=>'p2p_rescue','field'=>'slug','terms'=>['rescue-1','rescue-2'] ]]
    ]);
    return $q->posts ?: [];
  }

  /* Rescue Mode
   * Shortcode: [panic_to_plate_rescue_mode] (alias: [rescue_mode])
   */
  public function shortcode_rescue_mode($atts) {
    $out = '<div class="p2p-wrap"><div class="p2p-card">';
    $out .= '<h2>Rescue Mode</h2><p class="p2p-small">Pick what you have + your time. Get 3 ideas: fastest, healthiest, comfort-but-better.</p>';

    $out .= '<form method="post">';
    $out .= wp_nonce_field('p2p_rescue_mode', 'p2p_rm_nonce', true, false);

    $out .= '<div class="p2p-grid">';
    $out .= $this->select_field('rm_time', 'Time right now', [
      '' => 'Any',
      '5-10' => '5–10 min',
      '15-20' => '15–20 min',
      '30+' => '30+ min',
    ]);

    $out .= $this->select_field('rm_method', 'Equipment/method', [
      '' => 'Any',
      'microwave' => 'Microwave',
      'air-fryer' => 'Air fryer',
      'stovetop' => 'Stovetop',
      'no-cook' => 'No-cook',
      'one-pan' => 'One-pan',
    ]);
    $out .= '</div>';

    $pantry_terms = get_terms([
      'taxonomy' => 'p2p_pantry_item',
      'hide_empty' => false,
      'number' => 80,
    ]);

    $out .= '<div class="p2p-card">';
    $out .= '<h4>What do you have?</h4>';
    $out .= '<div class="p2p-grid-3">';
    if (!empty($pantry_terms) && !is_wp_error($pantry_terms)) {
      $selected = (array)($this->postv('pantry') ?? []);
      foreach ($pantry_terms as $t) {
        $checked = in_array($t->slug, $selected, true) ? 'checked' : '';
        $out .= '<label style="display:flex;gap:8px;align-items:center">';
        $out .= '<input type="checkbox" name="pantry[]" value="'.esc_attr($t->slug).'" '.$checked.'> '.esc_html($t->name);
        $out .= '</label>';
      }
    } else {
      $out .= '<p class="p2p-small">Add pantry items under Panic-to-Plate → Pantry Items (taxonomy terms).</p>';
    }
    $out .= '</div></div>';

    $out .= '<div class="p2p-actions p2p-no-print">';
    $out .= '<button class="p2p-btn" type="submit" name="p2p_rescue" value="1">Rescue me</button>';
    $out .= '</div>';

    $out .= '</form></div>';

    if (isset($_POST['p2p_rescue']) && $this->verify_nonce('p2p_rm_nonce', 'p2p_rescue_mode')) {
      $prefs = [
        'time' => sanitize_text_field($this->postv('rm_time')),
        'method' => sanitize_text_field($this->postv('rm_method')),
        'pantry' => array_map('sanitize_text_field', (array)$this->postv('pantry')),
      ];
      $out .= '<div class="p2p-card"><h3>Your Rescue Picks</h3>';
      $out .= $this->build_rescue_results($prefs);
      $out .= '</div>';
    }

    $out .= '</div>';
    return $out;
  }

  private function build_rescue_results($prefs) {
    $tax_query = ['relation' => 'AND'];
    $tax_query[] = ['taxonomy'=>'p2p_rescue','field'=>'slug','terms'=>['rescue-1','rescue-2']];

    if (!empty($prefs['time']))   $tax_query[] = ['taxonomy'=>'p2p_time','field'=>'slug','terms'=>[$prefs['time']]];
    if (!empty($prefs['method'])) $tax_query[] = ['taxonomy'=>'p2p_method','field'=>'slug','terms'=>[$prefs['method']]];

    if (!empty($prefs['pantry'])) {
      $tax_query[] = ['taxonomy'=>'p2p_pantry_item','field'=>'slug','terms'=>$prefs['pantry'],'operator'=>'IN'];
    }

    $q = new WP_Query([
      'post_type' => self::CPT,
      'post_status' => 'publish',
      'posts_per_page' => 12,
      'tax_query' => $tax_query,
      'orderby' => 'rand',
    ]);

    if (!$q->have_posts()) return '<p>No matches yet. Add more Rescue recipes or tag pantry items like “eggs”, “tortillas”, “rice”.</p>';

    $posts = $q->posts;
    shuffle($posts);
    $picks = array_slice($posts, 0, 3);

    $labels = ['Fastest', 'Healthiest', 'Comfort-but-better'];
    $html = '<div class="p2p-grid-3">';

    for ($i=0; $i<3; $i++) {
      $p = $picks[$i] ?? $picks[0];
      $html .= '<div class="p2p-card">';
      $html .= '<h4>'.esc_html($labels[$i]).'</h4>';
      $html .= '<p><a href="'.esc_url(get_permalink($p)).'">'.esc_html(get_the_title($p)).'</a></p>';
      $html .= '<div class="p2p-badges">'.$this->get_badges($p->ID).'</div>';

      $video = get_post_meta($p->ID, '_p2p_video_url', true);
      if ($video) $html .= '<p class="p2p-small"><a href="'.esc_url($video).'" target="_blank" rel="noopener">Watch video</a></p>';

      $html .= '</div>';
    }

    $html .= '</div>';
    return $html;
  }

  /* My Plans (History)
   * Shortcode: [panic_to_plate_my_plans]
   */
  public function shortcode_my_plans() {
    if (!is_user_logged_in()) {
      return '<div class="p2p-wrap"><div class="p2p-card"><h2>My Plans</h2><p>Please log in to view your saved plans.</p></div></div>';
    }

    $user_id = get_current_user_id();
    $history = get_user_meta($user_id, self::USER_META_HISTORY, true);
    if (!is_array($history) || empty($history)) {
      return '<div class="p2p-wrap"><div class="p2p-card"><h2>My Plans</h2><p>No saved plans yet. Generate your first plan in Autopilot.</p></div></div>';
    }

    $out = '<div class="p2p-wrap"><div class="p2p-card"><h2>My Plans</h2><p class="p2p-small">Your most recent plans are saved here.</p></div>';
    $nonce = wp_create_nonce('p2p_download');

    foreach ($history as $idx => $plan) {
      $created = isset($plan['created_at']) ? intval($plan['created_at']) : 0;
      $date = $created ? date_i18n(get_option('date_format').' '.get_option('time_format'), $created) : 'Unknown date';

      $dl = add_query_arg([
        'p2p_download_plan' => $idx,
        'p2p_type' => 'grocery',
        'p2p_nonce' => $nonce,
      ], home_url('/'));

      $out .= '<div class="p2p-card"><details class="p2p-details">';
      $out .= '<summary>Plan #'.($idx+1).' — '.$date.'</summary>';
      $out .= '<div class="p2p-hr"></div>';

      if (isset($plan['prefs']) && is_array($plan['prefs'])) {
        $p = $plan['prefs'];
        $out .= '<p class="p2p-small"><strong>Prefs:</strong> ';
        $out .= 'Cook nights: '.esc_html($p['cook_nights'] ?? '').' • ';
        $out .= 'Leftovers: '.esc_html($p['leftovers'] ?? '').' • ';
        $out .= 'Time: '.esc_html($p['time'] ?: 'any').' • ';
        $out .= 'Method: '.esc_html($p['method'] ?: 'any').' • ';
        $out .= 'Budget: '.esc_html($p['budget'] ?: 'any').' • ';
        $out .= 'Protein: '.esc_html($p['protein'] ?: 'any').'</p>';
      }

      $out .= '<div class="p2p-actions p2p-no-print">';
      $out .= '<button class="p2p-btn secondary" type="button" onclick="window.print()">Print</button>';
      $out .= '<a class="p2p-btn" href="'.esc_url($dl).'">Download grocery .txt</a>';
      $out .= '</div>';

      $items = isset($plan['grocery']) && is_array($plan['grocery']) ? $plan['grocery'] : [];
      sort($items);

      $out .= '<h4>Grocery List</h4><ul>';
      foreach ($items as $it) $out .= '<li>'.esc_html($it).'</li>';
      $out .= '</ul></details></div>';
    }

    $out .= '</div>';
    return $out;
  }

  /* Helpers */
  private function select_field($name, $label, $options, $default = '') {
    $val = esc_attr($this->postv_default($name, $default));
    $html = '<div class="p2p-field"><label>'.esc_html($label).'</label>';
    $html .= '<select name="'.esc_attr($name).'">';
    foreach ($options as $k => $v) {
      $sel = ((string)$k === (string)$val) ? 'selected' : '';
      $html .= '<option value="'.esc_attr($k).'" '.$sel.'>'.esc_html($v).'</option>';
    }
    $html .= '</select></div>';
    return $html;
  }

  private function postv($key) { return $_POST[$key] ?? ''; }

  private function postv_default($key, $default) {
    $v = $_POST[$key] ?? '';
    return ($v === '' || $v === null) ? $default : $v;
  }

  private function verify_nonce($field, $action) {
    return isset($_POST[$field]) && wp_verify_nonce($_POST[$field], $action);
  }
}

new P2P_Engine();
